# Copyright (c) 1994 by Sanjay Ghemawat
#############################################################################
#
# Exported Procedures
#
#	keymap_edit <title> <init-keymap> <var>
#		Allow user to edit <init-keymap>.  The final value
#		is stored in <var>.  Returns true iff user does not
#		cancel the edit.
#
#	key_shortform <seq>
#		Return short-form of key sequence
#
#	key_find_command <cmd> <keymap>
#		Return short-form of key sequence bound to <cmd>
#
#	keybindings_clear <widget>
#		Clear all key bindings in <widget>
#
#	keybindings_install <widget> <keymap> <command>
#		"Command" should be  a format string that contains
#		exactly one "%s".  For each entry "<key,action>" in
#		"keymap", bind "<key>" to "[format command <action>]"
#		in specified "widget".
#
# Lower-Level Exported Procedures
#
# keyentry <widget>			; Create key sequence entry widget
# keyentry_get <widget>			; Return current contents of keyentry
# keyentry_set <widget> <seq>		; Set keyentry contents
#
# keymap_editor <widget>		; Create key map editor widget
# keymap_get <widget>			; Get contents of key map editor
# keymap_set <widget> <keymap>		; Set contents of key map editor
# keymap_edit <title> <init> <var>	; Edit keymap

# effects - Clear all keybindings in widget
proc keybindings_clear {widget} {
    foreach b [bind $widget] {
	if [regexp Key- $b] {bind $widget $b ""}
    }
}

# requires - "command" is a format string that contains exactly one "%s" 
# effects  - Foreach entry "<key,action>" in "keymap", bind "<key>" to
#	     "[format command <action>]" in specified "widget".
proc keybindings_install {widget keymap command} {
    foreach b $keymap {
	bind $widget [lindex $b 0] [format $command [lindex $b 1]]
    }
}

# effects - Create key entry widget named $name.
proc keyentry {name} {
    global key

    set key($name,value) ""
    entry $name

    # Set-up key bindings for various modifier sequences
    bind $name <Key>		  [list keyentry_key $name <%K>]
    bind $name <Control-Key>	  [list keyentry_key $name <Control-%K>]
    bind $name <Meta-Key>	  [list keyentry_key $name <Meta-%K>]
    bind $name <Control-Meta-Key> [list keyentry_key $name <Control-Meta-%K>]
}

# effects - Return current contents of the key entry widget.
#	    The returned value can be passed directly to a "bind"
#	    command.  Note that the contents of the actual entry
#	    widget are a short representation of the actual key
#	    sequence and cannot be passed to a "bind" command.
proc keyentry_get {name} {
    global key
    return $key($name,value)
}

# effects - Set the contents of the entry widget.
proc keyentry_set {name value} {
    global key
    set key($name,value) $value
    keyentry_redisplay $name
}

# effects - Return short-form of specified key sequence.
#	    The returned value CANNOT be used in a "bind" command.
#	    It is only useful for presentation purposes.
proc key_shortform {seq} {
    regsub -all Control- $seq C-  seq
    regsub -all Meta-    $seq M-  seq
    regsub -all {><}	 $seq " " seq
    regsub -all {[<>]}   $seq "" seq
    return $seq
}

# A key map is specified as a list of pairs <key, string>.
# A "keymap_editor" widget can be used to edit such a map.

# effects Return short form of first binding for <command>
proc key_find_command {cmd keymap} {
    foreach b $keymap {
	if ![string compare [lindex $b 1] $cmd] {
	    return [key_shortform [lindex $b 0]]
	}
    }
    return ""
}

# effects Create a keymap editor widget named $w.
proc keymap_editor {w} {
    global key

    set key($w,rows) 0
    set key($w,columns) 3

    frame $w
    message $w.header -aspect 600 -padx 5m -pady 5m -relief raised -bd 1\
	-text [join {
	    {Edit key bindings by typing the actual keystrokes into the key}
	    {fields below and entering the corresponding action into the}
	    {values fields.  Use the "Clear" button to clear a key}
	    {sequence.  Use the "More Bindings" button to create blank key}
	    {bindings to fill in.}
	}]

    pack $w.header -side top -fill x -expand 1

    frame $w.values -bd 1 -relief raised
    pack $w.values -side top -fill both -expand 1

    _ke_more $w
}

# effects Get current bindings
proc keymap_get {w} {
    global key

    set result {}
    for {set r 0} {$r < $key($w,rows)} {incr r} {
	for {set c 0} {$c < 3} {incr c} {
	    set f $w.values.r$r.c$c
	    set k [keyentry_get $f.key]
	    set v [$f.val get]
	    if [string compare $k ""] {lappend result [list $k $v]}
	}
    }
    return [lsort -command _ke_cmp_by_key $result]
}

# effects Set bindings displayed in editor
proc keymap_set {w bindings} {
    set rows [expr ([llength $bindings]+2)/3]
    if {$rows < 1} {set rows 1}

    # Adjust displayed rows
    global key
    while {$rows > $key($w,rows)} {_ke_more $w}
    while {$key($w,rows) > $rows} {
	incr key($w,rows) -1
	destroy $w.values.r$key($w,rows)
    }

    for {set r 0} {$r < $key($w,rows)} {incr r} {
	for {set c 0} {$c < 3} {incr c} {
	    keyentry_set $w.values.r$r.c$c.key ""
	    $w.values.r$r.c$c.val delete 0 end
	}
    }

    set r 0
    set c 0
    foreach b $bindings {
	set f $w.values.r$r.c$c
	keyentry_set $f.key [lindex $b 0]
	$f.val delete 0 end
	$f.val insert insert [lindex $b 1]
	incr c
	if {$c == 3} {
	    set c 0
	    incr r
	}
    }
}

# Keymap dialog
#
# Commands
#
#	keymap_edit <title> <init-keymap> <var>
#		Allow user to edit <init-keymap>.  The final value
#		is stored in <var>.  Returns true iff user does not
#		cancel the edit.

proc keymap_edit {title init var} {
    set f .editkeys

    if ![winfo exists $f] {
	toplevel $f -class Dialog
	wm protocol $f WM_DELETE_WINDOW {set key(done) 0}

	keymap_editor $f.editor
	make_buttons $f.bot 3 {
	    {{More Bindings}	{_ke_more .editkeys.editor}}
	    {Clear		{_ke_clear .editkeys.editor}}
	    {Cancel		{set key(done) 0}}
	    {Okay		{set key(done) 1}}
	} 

	pack $f.editor -side top
	pack $f.bot -side bottom -fill x
	update idletasks
    } else {
	wm deiconify $f
    }

    # Fix dialog contents
    keymap_set $f.editor $init
    wm title $f $title

    # Run dialog
    global key
    set key(done) -1
    dialog_run "" $f $f.editor key(done)

    if $key(done) {
	upvar $var result
	set result [keymap_get $f.editor]
    }
    return $key(done)
}

#############################################################################
# Internal procedures

# effects Clear current focus entry (if any) in keymap.
proc _ke_clear {w} {
    set f [focus]
    if ![string match "$w.values.r*.c*.*" $f] {return}

    if [regexp {[.]values[.]r([0-9]+)[.]c([0-9])[.]key} $f j r c] {
	keyentry_set $w.values.r$r.c$c.key ""
    }

    if [regexp {[.]values[.]r([0-9]+)[.]c([0-9])[.]val} $f j r c] {
	$f delete 0 end
    }
}

# effects Add a few more bindings
proc _ke_more {w} {
    global key

    set r $w.values.r$key($w,rows)
    incr key($w,rows)
    frame $r
    pack $r -side top

    for {set col 0} {$col < $key($w,columns)} {incr col} {
	set c $r.c$col
	frame $c
	pack $c -side left -padx 5m

	keyentry $c.key
	entry $c.val
	$c.key configure -relief groove -bd 2 -width 11
	$c.val configure -relief groove -bd 2 -width 15

	pack $c.key -side left  -fill y -expand 1
	pack $c.val -side right -fill y -expand 1
    }
}

# comparison function for sorting a keymap by keys
proc _ke_cmp_by_key {a b} {
    return [string compare [lindex $a 0] [lindex $b 0]]
}

proc keyentry_redisplay {name} {
    global key
    $name delete 0 end
    $name insert insert [key_shortform $key($name,value)]
}

proc keyentry_key {name k} {
    # Ignore keys like Alt_L, Meta_R, ...
    if [regexp {_[LR]>$} $k] {return}

    global key
    set key($name,value) "$key($name,value)$k"
    keyentry_redisplay $name
}
