/* gitkeys.c -- an utility designed to help users to find out what is the 
   escape sequences sent by a particular key. Users can use this to set up
   their configuration files. */

/* Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Tudor Hulubei and Andrei Pitis.  */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#else /* !HAVE_STDLIB_H */
#include "ansi_stdlib.h"
#endif /* !HAVE_STDLIB_H */

#include <sys/types.h>

#ifdef HAVE_POSIXTTY
#include <termios.h>
#else
#ifdef HAVE_SYSTEMVTTY
#include <termio.h>
#else
#include <sgtty.h>
#endif /* HAVE_SYSTEMVTTY */
#endif /* HAVE_POSIXTTY */

#include <signal.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "stdc.h"
#include "xio.h"


#define TTY_OUTPUT 1


#ifdef HAVE_POSIXTTY
static struct termios old_term;
static struct termios new_term;
#else
#ifdef HAVE_SYSTEMVTTY
static struct termio old_term;
static struct termio new_term;
#else
static struct sgttyb  old_arg;
static struct tchars  old_targ;
static struct ltchars old_ltarg;
static struct sgttyb  new_arg;
static struct tchars  new_targ;
static struct ltchars new_ltarg;

/* NextStep doesn't define TILDE.  */
#ifndef TILDE
#define TILDE 0
#endif

#endif /* HAVE_SYSTEMVTTY */
#endif /* HAVE_POSIXTTY */


RETSIGTYPE do_exit __P((int));


void
tty_init()
{
#ifdef HAVE_POSIXTTY
    tcgetattr(TTY_OUTPUT, &old_term);

    signal(SIGTERM, do_exit);
    signal(SIGQUIT, do_exit);
    signal(SIGINT,  do_exit);

    new_term = old_term;
    new_term.c_iflag &= ~(ICRNL | IGNCR | INLCR | IGNBRK | BRKINT);
    new_term.c_oflag &= ~OPOST;
    new_term.c_lflag |= ISIG | NOFLSH;
    new_term.c_lflag &= ~(ICANON | ECHO);
    new_term.c_cc[VINTR] = 7;   /* ^G */
    new_term.c_cc[VQUIT] = 7;   /* ^G */
    new_term.c_cc[VMIN]  = 1;
    new_term.c_cc[VTIME] = 0;
    tcsetattr(TTY_OUTPUT, TCSADRAIN, &new_term);
#else
#ifdef HAVE_SYSTEMVTTY
    ioctl(TTY_OUTPUT, TCGETA, &old_term);

    signal(SIGTERM, do_exit);
    signal(SIGQUIT, do_exit);
    signal(SIGINT,  do_exit);

    new_term = old_term;
    new_term.c_iflag &= ~(ICRNL | IGNCR | INLCR);
    new_term.c_oflag = 0;
    new_term.c_lflag = 0;
    new_term.c_cc[VINTR] = 7;   /* ^G */
    new_term.c_cc[VQUIT] = 7;   /* ^G */
    new_term.c_cc[VMIN]  = 1;
    new_term.c_cc[VTIME] = 0;
    ioctl(TTY_OUTPUT, TCSETAW, &new_term);
#else
    ioctl(TTY_OUTPUT, TIOCGETP, &old_arg);
    ioctl(TTY_OUTPUT, TIOCGETC, &old_targ);
    ioctl(TTY_OUTPUT, TIOCGLTC, &old_ltarg);

    signal(SIGTERM, do_exit);
    signal(SIGQUIT, do_exit);
    signal(SIGINT,  do_exit);

    new_arg   = old_arg;
    new_targ  = old_targ;
    new_ltarg = old_ltarg;
    new_arg.sg_flags = ((old_arg.sg_flags &
                     ~(ECHO | CRMOD | XTABS | ALLDELAY | TILDE)) | CBREAK);
    new_targ.t_intrc   =  7;    /* ^G */
    new_targ.t_quitc   =  7;    /* ^G */
    new_targ.t_eofc    = -1;
    new_targ.t_brkc    = -1;
    new_ltarg.t_suspc  = -1;
    new_ltarg.t_dsuspc = -1;
    new_ltarg.t_rprntc = -1;
    new_ltarg.t_flushc = -1;
    new_ltarg.t_werasc = -1;
    new_ltarg.t_lnextc = -1;
    ioctl(TTY_OUTPUT, TIOCSETN, &new_arg);
    ioctl(TTY_OUTPUT, TIOCSETC, &new_targ);
    ioctl(TTY_OUTPUT, TIOCSLTC, &new_ltarg);
#endif /* HAVE_SYSTEMVTTY */
#endif /* HAVE_POSIXTTY */
}


void
tty_end()
{
#ifdef HAVE_POSIXTTY
    tcsetattr(TTY_OUTPUT, TCSADRAIN, &old_term);
#else
#ifdef HAVE_SYSTEMVTTY
    ioctl(TTY_OUTPUT, TCSETAW, &old_term);
#else
    ioctl(TTY_OUTPUT, TIOCSETN, &old_arg);
    ioctl(TTY_OUTPUT, TIOCSETC, &old_targ);
    ioctl(TTY_OUTPUT, TIOCSLTC, &old_ltarg);
#endif /* HAVE_SYSTEMVTTY */
#endif /* HAVE_POSIXTTY */
}


RETSIGTYPE
do_exit(signum)
    int signum;
{
    tty_end();
    exit(1);
}


int
main()
{
    char c;

#ifdef HAVE_GCC
    printf(PRODUCT" "VERSION" - Display key sequence utility\n");
#else
    printf("GNU Interactive Tools 4.3.7 - Display key sequence utility\n");
#endif /* !HAVE_GCC */
    printf("GIT is free software; you can redistribute it and/or modify it under the\n");
    printf("terms of the GNU General Public License as published by the Free Software\n");
    printf("Foundation; either version 2, or (at your option) any later version.\n");
    printf("Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.\n");
    printf("Written by Tudor Hulubei and Andrei Pitis, students at PUB, Romania\n");

    printf("\nPress space when done.\n\n");

    tty_init();

    for (;;)
    {
        read(0, &c, 1);

        if (c == ' ')
            break;

        printf("%x ", c);
        fflush(stdout);
    }

    tty_end();

    return 0;
}
